<?php
/**
 * Plugin Name: Folderit WP Embed
 * Description: Easily embed Folderit file listings into your WordPress posts and pages via shortcode.
 * Version: 0.9.6
 * Author: Folderit
 * Text Domain: folderit-wp-embed
 */

if ( ! defined( 'ABSPATH' ) ) exit;

// Register settings
function folderit_embed_register_settings() {
    add_option('folderit_client_id', '');
    add_option('folderit_client_secret', '');
    add_option('folderit_account_uid', '');
    add_option('folderit_cache_ttl', 300);
    register_setting('folderit_embed_options', 'folderit_client_id');
    register_setting('folderit_embed_options', 'folderit_client_secret');
    register_setting('folderit_embed_options', 'folderit_account_uid');
    register_setting('folderit_embed_options', 'folderit_cache_ttl');
}
add_action('admin_init', 'folderit_embed_register_settings');

// Settings page
add_action('admin_menu', function(){
    add_options_page('Folderit WP Embed Settings', 'Folderit WP Embed', 'manage_options', 'folderit-wp-embed', function(){
        ?>
        <div class="wrap">
          <h1>Folderit WP Embed Settings</h1>
          <form method="post" action="options.php">
            <?php settings_fields('folderit_embed_options'); ?>
            <table class="form-table">
              <?php foreach(['folderit_client_id'=>'Client ID','folderit_client_secret'=>'Client Secret','folderit_account_uid'=>'Account UID'] as $opt=>$label): ?>
              <tr><th><?php echo esc_html($label); ?></th>
                  <td><input name="<?php echo esc_attr($opt); ?>" value="<?php echo esc_attr(get_option($opt)); ?>" class="regular-text"></td></tr>
              <?php endforeach; ?>
              <tr><th>Cache TTL (seconds)</th>
                  <td><input type="number" name="folderit_cache_ttl" value="<?php echo esc_attr(get_option('folderit_cache_ttl')); ?>" class="small-text"></td></tr>
            </table>
            <?php submit_button('Save Changes'); ?>
          </form>
          <h2>Usage</h2>
          <p>Use this shortcode to list files only:</p>
          <pre style="background:#f1f1f1;padding:1rem;border-radius:4px;">
[folderit folder_uid="YOUR_FOLDER_UID" limit="20"]
          </pre>
        </div>
        <?php
    });
});

// OAuth2 token
function folderit_embed_get_token() {
    $cid = get_option('folderit_client_id');
    $cs  = get_option('folderit_client_secret');
    if ( ! $cid || ! $cs ) return '';
    if ( $t = get_transient('folderit_embed_token') ) return $t;
    $auth = base64_encode("$cid:$cs");
    $r = wp_remote_post('https://auth.folderit.com/oauth2/token', [
        'headers' => [
            'Authorization'=>'Basic '.$auth,
            'Content-Type'=>'application/x-www-form-urlencoded'
        ],
        'body' => ['grant_type'=>'client_credentials']
    ]);
    $d = json_decode(wp_remote_retrieve_body($r), true);
    if ( empty($d['access_token']) ) return '';
    set_transient('folderit_embed_token', $d['access_token'], get_option('folderit_cache_ttl'));
    return $d['access_token'];
}

// Fetch folder items
function folderit_embed_fetch_folder($uid, $limit) {
    $acc = get_option('folderit_account_uid');
    if ( ! $acc ) return [];
    $cache_key = 'folderit_folder_' . $uid;
    if ( $c = get_transient($cache_key) ) return $c;
    $token = folderit_embed_get_token();
    $resp = wp_remote_get("https://api.folderit.com/v2/accounts/$acc/entities/$uid/entities?per-page=$limit", [
        'headers'=>['Authorization'=>'Bearer ' . $token]
    ]);
    $items = json_decode(wp_remote_retrieve_body($resp), true);
    if ( ! is_array($items) ) return [];
    set_transient($cache_key, $items, get_option('folderit_cache_ttl'));
    return $items;
}

// Get download URL
function folderit_embed_get_download_url($fileUid) {
    $acc   = get_option('folderit_account_uid');
    $token = folderit_embed_get_token();
    $r = wp_remote_get("https://api.folderit.com/v2/accounts/$acc/files/$fileUid/download", [
        'headers'=>['Authorization'=>'Bearer ' . $token],
        'redirection'=>0
    ]);
    $h = wp_remote_retrieve_headers($r);
    return isset($h['location']) ? esc_url_raw($h['location']) : '';
}

// Shortcode rendering: files only
add_shortcode('folderit', function($atts){
    $a = shortcode_atts(['folder_uid'=>get_option('folderit_account_uid'),'limit'=>20], $atts);
    $items = folderit_embed_fetch_folder($a['folder_uid'], intval($a['limit']));
    if ( empty($items) ) return '<p>No files found.</p>';
    $html = "<ul class='folderit-list'>";
    foreach ( $items as $i ) {
        if ( ($i['type'] ?? '') === 'file' ) {
            $dl = folderit_embed_get_download_url($i['uid']);
            $name = esc_html($i['name'] ?? '');
            $html .= "<li>";
            $html .= "<span class='dashicons dashicons-admin-page'></span>";
            $html .= "<a href='{$dl}' download>{$name}</a>";
            $html .= "</li>";
        }
    }
    $html .= "</ul>";
    return $html;
});

// Enqueue CSS & Dashicons
add_action('wp_enqueue_scripts', function(){
    wp_enqueue_style('dashicons');
    wp_enqueue_style('folderit-wp-embed-css', plugin_dir_url(__FILE__).'assets/css/embed.css');
});
?>